/**
* \file         cbuffer.cpp
* \brief       circular buffer class implementation
* \author      Larry Salant
* \date        4/25/2008
*
*/
#include <string.h>
#include <stdlib.h>
#include "cbuffer.h"
#include "errors.h"
#include "os.h"

/**
* \author        Larry Salant
* \brief         constructor for circular buffer class
* \date          4/28/2008
* \param         sItems -number of items that circular buffer should hold
* \param         bWidth -number of bytes per item in buffer
* \return        void
* \retval        none
**/
CBuffer::CBuffer (UINT16 sItems, UINT8 bWidth):
  m_pBuffer(NULL),
  m_Size(0),
  m_ReadFrom(0),
  m_WriteTo(0),
  m_empty(TRUE),
  m_full(FALSE),
  m_fullcount(0),
  m_maxDepth(0)
{
//  m_ReadFrom = 0;
//  m_WriteTo = 0;

//  m_empty = TRUE;
//  m_full = FALSE;
  m_Size = sItems;
  // allocate a buffer for the specified number of items
  m_pBuffer = (UINT8 *)malloc(sItems*bWidth);
  DmFatalOnZeroPointer((void*)m_pBuffer, DM_ERROR_SUBSYS_GENERAL, ERR_GENERAL_ALLOC_MEMORY, 0);

  //Initialize debug variables
//  m_fullcount = 0;
//  m_maxDepth = 0;
}

CBuffer::~CBuffer()
{
    if (m_pBuffer)
    {
        free(m_pBuffer);
        m_pBuffer = NULL;
    }
}

/**
* \author        Larry Salant
* \brief         read a character from the circular buffer;  if buffer empty, will
*                record error - need to check if empty before calling
* \date          4/28/2008
* \param         none
* \return        UINT8
* \retval        character from buffer
**/
UINT8 CBuffer::Read(void)
{
  UINT8 bChar = 0;
  if (m_empty == TRUE)
    DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_GENERAL, ERR_GENERAL_CBUFFER_UNDERFLOW, 0);
  else
  {
    // this must be done without interruption.  cannot use a lock since it maybe called
    // from an ISR which cannot block
    OsEnterCritical();

    m_full = false; // buffer can't be m_full after a read
    bChar = m_pBuffer[m_ReadFrom];
    m_ReadFrom = (m_ReadFrom + 1) % m_Size;
    m_empty = (m_ReadFrom == m_WriteTo); // true if read catches up to write

    // exit the region
    OsExitCritical();

  }
  return bChar;
}
/**
* \author        Larry Salant
* \brief         copy from circular buffer to a linear buffer
* \date          4/28/2008
* \param         pBuffer - buffer to copy to
* \param         sMaxBytes - maximun number of bytes to copy
* \return        none
* \retval        void
**/
void CBuffer::CopyToLinear(UINT8 *pBuffer, UINT16 sMaxBytes)
{
  if (m_empty == TRUE)
    DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_GENERAL, ERR_GENERAL_CBUFFER_UNDERFLOW, 0);
  else
  {
    UINT16 sCount = 0;

    // this must be done without interruption.  cannot use a lock since it maybe called
    // from an ISR which cannot block
    OsEnterCritical();

    // copy from begining of circular buffer to linear buffer
    // until all bytes copied or Max bytes reached
    while (sCount <= sMaxBytes && !m_empty)
    {
      *pBuffer++ = m_pBuffer[m_ReadFrom];
      // increment pointer
      m_ReadFrom = (m_ReadFrom + 1) % m_Size;
      // empty if read catches up to write
      m_empty = (m_ReadFrom == m_WriteTo);
      sCount++;
    }
    m_full = false; // buffer can't be m_full after a read

    // exit the region
    OsExitCritical();
  }
}

/**
* \author        Larry Salant
* \brief         write a character to the circular buffer;  if buffer full, will
*                record error - need to check if full before calling
* \date          4/28/2008
* \param         bChar - byte to write
* \return        none
* \retval        void
**/
void CBuffer::Write(UINT8 bChar)
{
  if (m_full == TRUE)
    DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_GENERAL, ERR_GENERAL_CBUFFER_OVERFLOW, 0);
  else
  {
    // this must be done without interruption.  cannot use a lock since it maybe called
    // from an ISR which cannot block
    OsEnterCritical();

    m_pBuffer[m_WriteTo] = bChar;
    m_WriteTo = (m_WriteTo + 1) % m_Size;
    m_empty = false; // buffer can't be empty after a write
    m_full = (m_WriteTo == m_ReadFrom); // true if write catches up to read

    //Increment the full count if possible
    if ( m_full && (m_fullcount < 0xFF) )
      m_fullcount++;

    //Track the maximum depth
    INT16 depth = (m_WriteTo - m_ReadFrom);
    if (depth < 0)
      depth = m_Size + depth;
    if (depth > m_maxDepth)
      m_maxDepth = depth;

    // exit the region
    OsExitCritical();
  }
}
/**
* \author        Larry Salant
* \brief         read a long word from the circular buffer;  if buffer empty, will
*                record error - need to check if empty before calling
* \date          4/28/2008
* \param         none
* \return        UINT32
* \retval        character from buffer
**/
UINT32 CBuffer::ReadLong(void)
{
  UINT32 lItem = 0;

  if (m_empty == TRUE)
    DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_GENERAL, ERR_GENERAL_CBUFFER_UNDERFLOW, 0);
  else
  {
    UINT32 *pLongBuf = (UINT32 *)m_pBuffer;
    // this must be done without interruption.  cannot use a lock since it maybe called
    // from an ISR which cannot block
    OsEnterCritical();

    lItem = pLongBuf[m_ReadFrom];
    m_ReadFrom = (m_ReadFrom + 1) % m_Size;
    m_empty = (m_ReadFrom == m_WriteTo); // true if read catches up to write
    m_full = false; // buffer can't be m_full after a read

    // exit the region
    OsExitCritical();
  }
  return lItem;
}

/**
* \author        Larry Salant
* \brief         write a long word to the circular buffer;  if buffer full, will
*                record error - need to check if full before calling
* \date          4/28/2008
* \param         bChar - byte to write
* \return        none
* \retval        void
**/
void CBuffer::WriteLong(UINT32 lItem)
{
  if (m_full == TRUE)
    DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_GENERAL, ERR_GENERAL_CBUFFER_OVERFLOW, 0);
  else
  {
    UINT32 *pLongBuf = (UINT32 *)m_pBuffer;

    // this must be done without interruption.  cannot use a lock since it maybe called
    // from an ISR which cannot block
    OsEnterCritical();

    pLongBuf[m_WriteTo] = lItem;
    m_WriteTo = (m_WriteTo + 1) % m_Size;
    m_full = (m_WriteTo == m_ReadFrom); // true if write catches up to read
    m_empty = false; // buffer can't be empty after a write

    // for tuning, keep track of how many times queue is filled
    // (possibly should make this max count in queue)
    if ( m_full && (m_fullcount < 0xFF) )
      m_fullcount++;

    //Track the maximum depth
    INT16 depth = (m_WriteTo - m_ReadFrom);
    if (depth < 0)
      depth = m_Size + depth;
    if (depth > m_maxDepth)
      m_maxDepth = depth;

    // exit the region
    OsExitCritical();
  }
}
/**
* \author        Larry Salant
* \brief         checks if circular buffer is empty
* \date          4/28/2008
* \param         none
* \return        BOOL
* \retval        TRUE if empty (otherwise 0)
**/
BOOL CBuffer::IsEmpty(void)
{
  return m_empty;
}
/**
* \author        Larry Salant
* \brief         checks if circular buffer is full
* \date          4/28/2008
* \param         none
* \return        BOOL
* \retval        TRUE if full (otherwise 0)
**/
BOOL CBuffer::IsFull(void)
{
  return m_full;
}

/**
* \author        Adolfo Velasco
* \brief         Gets the full count
* \date          06/18/2009
* \param         none
* \return        UINT8
* \retval
**/
UINT8 CBuffer::GetFullCount(void)
{
    return m_fullcount;
}

/**
* \author        Adolfo Velasco
* \brief         Gets the max depth
* \date          06/18/2009
* \param         none
* \return        UINT8
* \retval
**/
INT16 CBuffer::GetMaxDepth(void)
{
    return m_maxDepth;
}



